﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using BSDKW;
using Urbex.Controls;
using System.IO;

namespace SampleMapApplication
{
    public partial class Form1 : Form
    {
        BDataLoader bDataLoader;
        MapSnapshot mapSnapShot;
        Map map;

        public Form1()
        {
            InitializeComponent();
            AddMapControl();
        }

        /// <summary>
        /// Adds a map.net control to the application.
        /// </summary>
        private void AddMapControl()
        {
            //First initialize the BSDK for use.
            BSDK.Initialize(0, 0, null);

            //Create a dataloader, which is used by map to fetch data.
            bDataLoader = BSDK.CreateNewDataLoader();

            //Ask the user for the Urbex server's data credentials.
            Credentials credentials = new Credentials(bDataLoader);

            //If user have given correct and authenticated credentials, then load the map control; else exit the application.
            if (credentials.ShowDialog() == DialogResult.OK)
            {
                //Initialize the Map initial options through a MapSnapshot
                mapSnapShot = new MapSnapshot();

                //Set the cross hair style which is shown in the center of the screen.
                mapSnapShot.CrossHairStyle = new CrossHairStyle(true, Color.Yellow, CrossHairSymbol.BigWindow);

                //Set the view and zoom level of the map.
                mapSnapShot.MapViewType = ViewType.Mosaic;
                mapSnapShot.Zoom = 15;

                //Sets the default longitute and latitude.
                mapSnapShot.CenterPoint = new XY(bDataLoader.DefaultLongitude, bDataLoader.DefaultLatitude);

                //Initialize the Map object.
                map = new Map(bDataLoader, mapSnapShot);

                map.ForceToDisplayNorthSideUp(true);

                //This loads the zoombar and orientation map sprites.
                map.LoadControlsSet(MapControls.Navigation);

                //Pans to the default longitute and latitude.
                //map.PanToLatLon(bDataLoader.DefaultLongitude, bDataLoader.DefaultLatitude);

                if (bDataLoader.DefaultLongitude == 0 && bDataLoader.DefaultLatitude == 0)
                    map.SetOrthoZoom(3, false);

                map.Size = MapPanel.Size;
                map.Anchor = AnchorStyles.Left | AnchorStyles.Bottom | AnchorStyles.Right | AnchorStyles.Top;
                //Add the map control to the map panel.
                this.MapPanel.Controls.Add(map);

                //Add a mossaic imagery layer which contains the map images.
                MossaicImageryLayer oMossaicImageryLayer = new MossaicImageryLayer(map.MapContext, bDataLoader, true, 
                    OrientationType.Ortho, null, null, ProjectionOsr.SphericalMercator.Name);
                map.AddOverlay(oMossaicImageryLayer);
            }
            else
            {
                Application.Exit();
            }
        }

        /// <summary>
        /// Sets latitude and longitude for location London.
        /// </summary>
        private void GotoLondon_Click(object sender, EventArgs e)
        {
            map.PanToLatLon(-0.12766, 51.507276);
        }

        /// <summary>
        /// Sets latitude and longitude for location Paris.
        /// </summary>
        private void GotoParis_Click(object sender, EventArgs e)
        {
            map.PanToLatLon(2.465163, 48.817707);
        }

        /// <summary>
        /// Opens a shape file and adds it to the map.
        /// </summary>
        private void AddShape_Click(object sender, EventArgs e)
        {
            OpenFileDialog dlg = new OpenFileDialog();

            dlg.Multiselect = false;
            // Set filter for shape files 
            dlg.DefaultExt = "*.shp";
            StringBuilder Filter = new StringBuilder();

            Filter.AppendFormat("Shapefiles (*.shp)|*.shp");
            dlg.Filter = Filter.ToString();

            //If user have selected some thing then open the file.
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                //Get the .prj file for the shape file. The .prj file contains the projection on which the shape should be shown.
                string filename = System.IO.Path.GetFileNameWithoutExtension(dlg.FileName);
                string extension = System.IO.Path.GetExtension(dlg.FileName);
                ProjectionOsr projection = null;

                string ProjectionFilePath = System.IO.Path.GetDirectoryName(dlg.FileName) + @"\" + filename + ".prj";
                FileInfo ProjectionFile = new FileInfo(ProjectionFilePath);

                if (ProjectionFile.Exists)
                {
                    //If projection file exists then try to find the projection from it.
                    projection = ProjectionsLookup.LookupUsingPRJFile(ProjectionFilePath);

                    if (projection != null)
                    {
                        //If projection found then add a vector layer.
                        VectorLayer oVectorLayer = new VectorLayer(map, false);

                        //Create a object of vector file reader which reads the shape file.
                        VectorFileReader vectorFileReader = new VectorFileReader(dlg.FileName, projection);

                        //Finally read the shape file and add the content to the vector layer.
                        oVectorLayer.OpenVectorFile(vectorFileReader);

                        //Add the vector layer to the map.
                        map.AddOverlay(oVectorLayer);

                        //Fit the vector layer to the screen so that user can see the added shape file.
                        oVectorLayer.FitToScreen();
                    }
                    else
                    {
                        MessageBox.Show("Invalid projection detected. Please provide a valid .prj file.");
                    }
                }
                else
                {
                    MessageBox.Show("Projection for the shape file could not be found. The projections are stored in the .prj file.");
                }
            }
        }
    }
}
